<?php

namespace App\Imports;

use App\Models\Inventory;
use Illuminate\Support\Collection; // Penting: Ini untuk ToCollection
use Maatwebsite\Excel\Concerns\ToCollection; // Ubah dari ToModel ke ToCollection
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithValidation;
use Maatwebsite\Excel\Concerns\WithBatchInserts; // Untuk performa
use Maatwebsite\Excel\Concerns\WithChunkReading; // Untuk performa
use Illuminate\Support\Facades\Log; // Untuk logging
use Carbon\Carbon; // Untuk parsing tanggal

class InventoriesImport implements ToCollection, WithHeadingRow, WithValidation, WithBatchInserts, WithChunkReading
{
    private $importedCount = 0; // Menghitung item yang berhasil diimpor

    /**
     * @param Collection $rows
     */
    public function collection(Collection $rows)
    {
        foreach ($rows as $row) {
            // Bersihkan nilai 'nan' atau string kosong menjadi null
            $row = $row->map(function($value) {
                return (is_string($value) && (strtolower($value) == 'nan' || $value == '')) ? null : $value;
            });

            // Ambil data dari header Excel yang baru Anda tentukan
            $nama_barang = trim($row['nama_barang'] ?? '');
            $merek_model = trim($row['merek_model'] ?? '');
            $tahun_perolehan_str = trim($row['tahun_perolehan'] ?? '');
            $keterangan_umum = trim($row['keterangan'] ?? '');

            $jumlah_baik = (int)($row['jumlah_total_baik'] ?? 0);
            $jumlah_rusak_ringan = (int)($row['jumlah_total_rusak_ringan'] ?? 0);
            $jumlah_rusak_berat = (int)($row['jumlah_total_rusak_berat'] ?? 0);

            $jumlah_total_row = $jumlah_baik + $jumlah_rusak_ringan + $jumlah_rusak_berat;

            if (empty($nama_barang) || $jumlah_total_row == 0) {
                Log::warning("Skipping row due to missing 'Nama Barang' or 'Jumlah Total' is zero for row: " . json_encode($row->toArray()));
                continue; // Lewati baris yang tidak valid
            }

            // --- Pemrosesan purchase_date ---
            $purchase_date = null;
            if (!empty($tahun_perolehan_str)) {
                try {
                    // Jika hanya tahun, asumsikan 1 Januari
                    if (preg_match('/^\d{4}$/', $tahun_perolehan_str)) {
                        $purchase_date = $tahun_perolehan_str . '-01-01';
                    } else {
                        // Coba parse sebagai tanggal lengkap jika ada
                        $purchase_date = Carbon::parse($tahun_perolehan_str)->format('Y-m-d');
                    }
                } catch (\Exception $e) {
                    Log::warning("Invalid purchase_date format for '{$nama_barang}': {$tahun_perolehan_str}. Error: " . $e->getMessage());
                }
            }

            // --- Field umum untuk semua unit dari baris Excel ini ---
            $common_fields = [
                'description' => trim("Nama: {$nama_barang}, Merek/Model: {$merek_model}, Tahun Perolehan: {$tahun_perolehan_str}. Keterangan: {$keterangan_umum}"),
                'location' => $row['location'] ?? 'Lab TJKT', // Asumsi lokasi default atau bisa ditambahkan kolom Excel
                'owner' => $row['owner'] ?? 'Sekolah', // Asumsi kepemilikan default atau bisa ditambahkan kolom Excel
                'notes' => $keterangan_umum,
                'purchase_date' => $purchase_date,
            ];

            $unit_counter_for_row = 1;
            $current_year_two_digits = date('y');

            // Kondisi yang akan diimpor (sesuai DB)
            $conditions_to_process = [
                'Baik' => $jumlah_baik,
                'Rusak Ringan' => $jumlah_rusak_ringan, // Dari 'Jumlah Total (Rusak Ringan)'
                'Rusak Berat' => $jumlah_rusak_berat, // Dari 'Jumlah Total (Rusak Berat)'
            ];

            // Jika total jumlah dari kondisi adalah 0 tapi jumlah total baris lebih dari 0,
            // asumsikan semua unit adalah 'Baik'.
            if (array_sum($conditions_to_process) == 0 && $jumlah_total_row > 0) {
                $conditions_to_process['Baik'] = $jumlah_total_row;
            }


            foreach ($conditions_to_process as $db_condition_name => $count_for_condition) {
                for ($i = 0; $i < $count_for_condition; $i++) {
                    // Membuat string yang bersih untuk code dan serial_number
                    $clean_name_for_id = substr(str_replace([' ', '/', '.', '-'], '', strtoupper($nama_barang)), 0, 15);
                    $clean_merk_model_for_id = substr(str_replace([' ', '/', '.', '-'], '', strtoupper($merek_model)), 0, 10);
                    $clean_condition_short = substr(str_replace(' ', '', $db_condition_name), 0, 5);

                    // Buat code unik
                    $item_code = "INV-{$clean_name_for_id}-{$clean_merk_model_for_id}-{$clean_condition_short}-{$unit_counter_for_row}-{$current_year_two_digits}";
                    $item_code = preg_replace('/[^A-Z0-9-]/', '', $item_code);
                    $item_code = preg_replace('/-+/', '-', $item_code);
                    $item_code = trim($item_code, '-');
                    $item_code = substr($item_code, 0, 250); // Batasi panjang kolom database

                    // Buat serial_number unik
                    $serial_number = "SN-{$clean_name_for_id}-{$clean_merk_model_for_id}-{$current_year_two_digits}-{$clean_condition_short[0]}-{$unit_counter_for_row}";
                    $serial_number = preg_replace('/[^A-Z0-9-]/', '', $serial_number);
                    $serial_number = preg_replace('/-+/', '-', $serial_number);
                    $serial_number = trim($serial_number, '-');
                    $serial_number = substr($serial_number, 0, 250); // Batasi panjang kolom database

                    try {
                        // Coba temukan item berdasarkan code, jika ditemukan, update. Jika tidak, buat baru.
                        $inventory = Inventory::firstOrNew(['code' => $item_code]);

                        // Jika item sudah ada dan serial_number yang digenerate berbeda, itu berarti ini mungkin unit baru.
                        // Jika serial_number yang digenerate ternyata sudah ada di item lain, akan ada QueryException.
                        // Untuk kasus konflik SN, kita log dan lewati unit ini.
                        if ($inventory->exists && !empty($serial_number) && $inventory->serial_number !== $serial_number) {
                            // Jika ada konflik SN untuk kode yang sama, kita bisa coba generate kode/SN baru
                            // atau menganggap ini sebagai duplikasi dan melewati. Untuk saat ini, log warning.
                            Log::warning("Code '{$item_code}' exists, but generated SN '{$serial_number}' conflicts or is new. Attempting update with new SN.");
                        }

                        $inventory->name = trim("{$nama_barang} {$merek_model}");
                        $inventory->code = $item_code;
                        $inventory->serial_number = $serial_number; // Serial number yang digenerate
                        $inventory->description = $common_fields['description'];
                        $inventory->location = $common_fields['location'];
                        $inventory->condition = $db_condition_name;
                        $inventory->purchase_date = $common_fields['purchase_date'];
                        $inventory->owner = $common_fields['owner'];
                        $inventory->notes = $common_fields['notes'];

                        $inventory->save();
                        $this->importedCount++;

                    } catch (\Illuminate\Database\QueryException $e) {
                        if (str_contains($e->getMessage(), 'Duplicate entry')) {
                            $error_message = $e->getMessage();
                            if (str_contains($error_message, "'inventories_code_unique'")) {
                                Log::warning("Duplicate code '{$item_code}' during import. Attempting to update existing item with this code.");
                                // Jika kode duplikat, berarti ini update item yang sudah ada. Coba update saja.
                                // Logic firstOrNew sudah mencoba mencari berdasarkan code, jadi ini hanya konfirmasi
                            } elseif (str_contains($error_message, "'inventories_serial_number_unique'")) {
                                Log::warning("Duplicate serial_number '{$serial_number}' generated for item code '{$item_code}'. Skipping this unit to prevent SN conflict.");
                                // Lewati unit ini karena SN yang digenerate konflik dengan item lain yang mungkin sudah ada.
                            } else {
                                Log::error("Unhandled Duplicate entry error for '{$item_code}': " . $error_message);
                                throw $e; // Lempar error lain
                            }
                        } else {
                            Log::error("Error importing record for '{$item_code}': " . $e->getMessage() . " on row " . ($row->get('row_number') ?? 'unknown') . ". Full Row: " . json_encode($row->toArray()));
                            throw $e; // Lempar error lain
                        }
                    }
                    $unit_counter_for_row++;
                }
            }
        }
    }

    /**
     * Definisi aturan validasi untuk setiap kolom Excel.
     * Dengan WithHeadingRow, nama kolom akan di-snake_case dan lowercase otomatis.
     */
    public function rules(): array
    {
        return [
            'nama_barang' => 'required|string|max:255',
            'merek_model' => 'nullable|string|max:255',
            'tahun_perolehan' => 'nullable|string|max:255', // Biarkan sebagai string, kita parse di logika
            'keterangan' => 'nullable|string|max:65535',
            'jumlah_total_baik' => 'nullable|integer',
            'jumlah_total_rusak_ringan' => 'nullable|integer',
            'jumlah_total_rusak_berat' => 'nullable|integer',
            // Kolom lain seperti location, owner, condition, purchase_date, code, serial_number
            // akan diisi atau divalidasi dalam logika `collection`
        ];
    }

    /**
     * (Opsional) Sesuaikan pesan validasi.
     */
    public function customValidationMessages()
    {
        return [
            'nama_barang.required' => 'Kolom "Nama Barang" wajib diisi di Excel.',
            'jumlah_total_baik.integer' => 'Kolom "Jumlah Total (Baik)" harus angka.',
            'jumlah_total_rusak_ringan.integer' => 'Kolom "Jumlah Total (Rusak Ringan)" harus angka.',
            'jumlah_total_rusak_berat.integer' => 'Kolom "Jumlah Total (Rusak Berat)" harus angka.',
        ];
    }

    public function batchSize(): int
    {
        return 500; // Ukuran batch untuk insert (sesuaikan dengan performa server)
    }

    public function chunkSize(): int
    {
        return 500; // Ukuran chunk untuk membaca (sesuaikan dengan memori)
    }

    public function getImportedCount(): int
    {
        return $this->importedCount;
    }
}